/*****************************************************************************
*  LPC54xx  PWM deadtime  program
*
*  Use SCT timer to generate a two-channel double-edge controlled PWM
*  intended for use as a complementary PWM pair with dead-time control
*****************************************************************************/

#include "board.h"


/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
#define DC1        (130)                                   			 	// duty cycle 1
#define DC2        (135)                                   			 	// duty cycle 2
#define hperiod    (180)
/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/
void SCT_Init(void);
/*****************************************************************************
 * Public functions
 ****************************************************************************/

void SCT_Init(void)
{
	Chip_SCT_Init(LPC_SCT);			                   			 								// enable the SCT1 clock

	Chip_SCT_Config(LPC_SCT, SCT_CONFIG_16BIT_COUNTER      |			 						// split timers, auto limit
                                  SCT_CONFIG_AUTOLIMIT_L   );

	Chip_SCT_SetControl(LPC_SCT, SCT_CTRL_BIDIR_L(1));     		 								// configure SCT1 as BIDIR

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_0, hperiod);			 							// match on (half) PWM period
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, hperiod);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_1, DC1);				 							// match on duty cycle 1
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1, DC1);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_2, DC2);				 							// match on duty cycle 2
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_2, DC2);

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0, ENABLE_ALL_STATES);	 							// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0,(CHIP_SCT_EVENTCTRL_T)  ( SCT_IOCOND_FALL   |
                                                                       SCT_IOSEL_1 	     |    // SCT0_IN1 select
                                                                       SCT_COMBMODE_IO ));  	// IN_1 falling edge only condition

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1, ENABLE_ALL_STATES);	 							// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1,(CHIP_SCT_EVENTCTRL_T)  ( SCT_IOCOND_RAISE |
                                                                       SCT_IOSEL_1	    |     // SCT0_IN1 select
                                                                       SCT_COMBMODE_IO  ));  	// IN_1 rising edge only condition

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_2, ENABLE_ALL_STATES);	 							// event 2 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_2,(CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1  |
                                                                       SCT_IOSEL_1		     |   // SCT0_IN1 select
                                                                       SCT_COMBMODE_MATCH  )); // match 1 (DC1) only condition

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_3, ENABLE_ALL_STATES);	 							// event 3 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_3,(CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2  |
                                                                       SCT_IOSEL_1		     |   // SCT0_IN1 select
                                                                       SCT_COMBMODE_MATCH  )); // match 2 (DC) only condition


	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_4, (CHIP_SCT_EVENT_T) ( SCT_EVT_0 |
                                                                   SCT_EVT_2 ));  				// event 0 and 2 set OUT4 (blue LED)

	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_4, SCT_EVT_2);		    							// event 2 clears OUT4 (blue LED)

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_5, SCT_EVT_3);			    						// event 3 sets OUT5 (green LED)

	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_5, (CHIP_SCT_EVENT_T) ( SCT_EVT_0 |
                                                                     SCT_EVT_3 ));				// event 0 and 3 clear OUT5 (green LED)

	Chip_SCT_SetConflictResolution(LPC_SCT, 4, 0x03);  											// toggle OUT4 on conflict

	Chip_SCT_SetConflictResolution(LPC_SCT, 5, 0x03);  											// toggle OUT5 on conflict

	Chip_SCT_Output(LPC_SCT, 0x30 );                   											// default set OUT4 and and clear OUT5

    LPC_SCT->STOP_L  = SCT_EVE_0 ;                												// event 0 will stop the timer

    Chip_SCT_EnableEventInt(LPC_SCT,SCT_EVT_1);           										// event 1 will generate an irq

    NVIC_EnableIRQ(SCT0_IRQn);                             										// enable SCT1 interrupt

    Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);	   											// unhalt it by clearing bit 2 of the CTRL register
}

void SCT0_IRQHandler(void)
{

	Chip_SCT_SetControl(LPC_SCT, SCT_CTRL_CLRCTR_L);			// clear the L counter
	Chip_SCT_ClearControl(LPC_SCT, SCT_CTRL_STOP_L);          // start the L counter

	Chip_SCT_SetEventFlag(LPC_SCT, SCT_EVT_1);			// clear event 1 interrupt flag
}


int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();




	Chip_Clock_EnablePeriphClock(	(CHIP_SYSCON_CLOCK_T) ( SYSCON_CLOCK_GPIO0	  | 		// enable GPIO0 clock
                                                        SYSCON_CLOCK_GPIO1	  | 		// enable GPIO1 clock
                                                        SYSCON_CLOCK_INPUTMUX 	    |			  // enable PMUX clock
                                                        SYSCON_CLOCK_IOCON	));		  //enable IOCON clock

	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 2, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);		//SCT0_OUT5 = P1.2  (green   LED)
	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 1, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);		//SCT0_OUT4 = P1.1  (blue   LED)


	Chip_GPIO_SetPinDIRInput(LPC_GPIO, 0, 24);


	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 24, 	IOCON_FUNC0 		  |
                                          IOCON_I2C_SLEW		|
                                          IOCON_DIGITAL_EN	|
                                          IOCON_INPFILT_OFF	|
                                          0x0 << 6			    |
                                          0x0 << 9			    |
                                          0x1 << 10			    );


    SCT_Init();                                          			// Initialize SCT0


    while (1)                                              			// loop forever
    {
    	__WFI();
    }
}
